
###################################################
# データ構造：ベクトル
###################################################

# ベクトルとは「スカラーを複数並べたデータ構造」のこと
# Rオブジェクトは基本的にはベクトルとして扱われる
# (スカラーは長さ1のベクトルとして扱い) 
# ベクトルの各要素は同じデータ型でなければならない
# (異なる場合は片方に統一される)

### 関数cの使い方
(x <- c(0,1,2,3,4)) # スカラーを並べてベクトルを作成
(y <- c("foo","bar")) # 文字列のベクトル
x[2] # ベクトルの2番目の要素
y[2]
x[c(1,3,5)] # 複数の要素は要素番号のベクトルで指定

### 関数seqの使い方
(x <- seq(0,3,by=0.5)) # 0から3まで0.5刻みの系列
(y <- seq(0,3,length=5)) # 0から3まで長さ5の系列
(z <- 1:10) # seq(1,10,by=1)と同様
(z <- 10:1) # 10から1まで1刻みの逆順
z[3:8] # zの3番目から8番目の要素

### 関数repの使い方
(x <- rep(1,7)) # 1を7回繰り返す
(y <- rep(c(1,2,3),times=3)) # (1,2,3)を3回繰り返す
(z <- rep(c(1,2,3),each=3)) # (1,2,3)をそれぞれ3回繰り返す

### その他の便利な操作
(x <- seq(0,2,by=0.3))
length(x) # ベクトルの長さ
y <- 2:5
(z <- c(x,y)) # ベクトルの連結
rev(z) # rev はベクトルを反転する関数
sum(z) # 総和
diff(z) # 差分
mean(z) # 平均
sd(z) # 標準偏差
max(z) # 最大値
min(z) # 最小値

### ベクトルの計算
a <- 1:3
b <- 4:6
a + b # 足し算
a + 1:6 # 長さが異なる場合は足りない方が周期的に拡張される
a + 1:5 # 一方の長さがもう一方の長さの整数倍出ない場合は警告
a - b # もちろん引き算も可
2 * a # スカラー倍
a * b　# 成分ごとの掛け算
a * 1:6 # 長さが異なる場合は足りない方が周期的に拡張される
a/b # 除算も成分ごと
a/2 # aの全ての成分を2で割る
sin(pi*a/2) # 初等関数は成分ごとに適用される

### ベクトルの成分の抽出
x <- c(4, 1, 2, 9, 8, 3, 6)
x[c(5, 2)] # 5番目と2番目の要素をこの順で抽出
x[-c(2, 3, 7)] # 2,3,7番目以外の要素を表示
(idx <- x > 3) # 3より大きい要素はTRUE, 3以下の要素はFALSE
x[idx] # 3より大きい要素をすべて表示
x[x > 3] # 上と同じ
x[-c(2, 3, 7)] # 2,3,7番目以外の要素を表示
x[c(2, 5)] <- c(0, 1) # 2番目と5番目の要素を文字0と1に置換
x
(names(x) <- c("a", "b", "c", "d", "e", "f", "g")) # xの成分にアルファベット順に名前をつける
x[c("b", "e")] # 2番目と5番目の要素

#################################################
# データ構造：行列
#################################################

# 行列とは, 「スカラーを長方形状に並べたデータ構造」のこと
# Rでは,「行の数」と「列の数」という属性(次元属性)をもつ
# ベクトルとして扱われる

### 関数matrixの使い方
x <- c(2, 3, 5, 7, 11, 13)
matrix(x, 2, 3) # (2,3)行列
(X <- matrix(x, ncol = 3)) # 列数指定(行数はそれに合わせて決まる)
(Y <- matrix(x, ncol = 3, byrow = TRUE)) # 横に並べる
# その他の操作
nrow(X) # 行数
ncol(X) # 列数
X[1, 2] # (1,2)成分
X[2, ] # 2行目
X[, 3] # 3列目
as.vector(X) # ベクトルxに戻る
dim(x) <- c(2, 3) # ベクトルに次元属性を与えて行列化
x # 行列Xになる

### 行列の計算
X + Y # 足し算
X - Y # 引き算
t(X) # 転置
X + t(X) # サイズが合わないとエラー
X * Y # 成分ごとの積(アダマール積)
t(X) %*% Y # 通常の意味の行列の積
crossprod(X, Y) # 上と同じ計算をする
X %*% Y # サイズが合わないとエラー
X / Y # 成分ごとの割り算

#################################################
# データ構造：リスト
#################################################

# リストとは, ベクトルを一般化し, 異なる構造のデータを
# まとめて1つのオブジェクトとして扱えるようにしたもの
# リストの各要素はデータ型・クラスともにバラバラであってもよい

### 関数listの使い方
# 各成分のデータ型はバラバラでよい
(L1 <- list(c(1, 2, 5, 4), matrix(1:4, 2), c("Hello", "World")))
str(L1) # データの構造
L1[[1]] # L1の第1成分の取り出し
L1[[2]][2, 1] # 2番目の要素の(2,1)成分
L1[[c(3, 2)]] # 3番目の要素の2番目
L1[[3]][[2]] # 上と同じ
L1[c(1, 3)] # 複数成分を同時に抽出
L1[1] # 第1成分をリストとして取り出す
(L2 <- list(Info = "統計データ解析", List1 = L1)) # 名前付きリスト生成
L2[["Info"]] # 要素名で取り出し
L2$Info # 別記法

names(L1) <- c("vector", "matrix", "character") # L1の要素に名前をつける
L1

#################################################
#  乱数
#################################################

# 乱数とはランダムに生成された数列のこと
# 統計学やデータサイエンスにおける数値シミュレーションに
# おいて重要な役割を果たす

### 関数rnormの使い方(正規分布に従う乱数の発生)
rnorm(5,mean=3,sd=2) # 平均3, 標準偏差2の正規乱数を5個生成
rnorm(5) # 平均と標準偏差について指示がなければ標準正規分布

# 数値シミュレーションの再現可能性を担保するために, 
# 乱数を利用する際には初期値を指定することが多い
# (同一の初期値から生成される乱数は同一のものとなる)
# Rでは関数set.seedによって初期値を指定できる

### 関数set.seedについて
set.seed(1) # 乱数の初期値をseed=1で指定
rnorm(5) 
set.seed(2) # 乱数の初期値をseed=2で指定
rnorm(5) # seed=1の場合と異なる結果となる
set.seed(1) # 乱数の初期値をseed=1で指定
rnorm(5) # 初めのseed=1の結果と同じ

#################################################
#  グラフィックス
#################################################

### 基本的な描画
theta <- seq(0, 2*pi, length = 100)
x <- sin(theta)
plot(x) # 横軸に成分番号, 縦軸にxの各成分をプロット
plot(theta, # 横軸をthetaにしてみる
     x, 
     xlab = expression(theta), # x軸のラベル
     ylab = expression(sin(theta))) # y軸のラベル
plot(theta, x, type = "l", col = "red") # 赤色で折れ線プロット
plot(sin, 0, 2*pi, lty = "dotted") # 関数を定義域を定めて描画
plot(sin(theta), cos(theta), # 単位円
     main = "circle", # タイトル
     type = "l", lwd = 3) # lwdは線の太さ
grid(col = "darkgray") # グリッドを追加
abline(0, 1, col = "green", lty = "dashed") # 切片0, 傾き1の直線を追加

### ヒストグラム
## 正規分布
z <- rnorm(10000, 1, 1) # 平均1, 標準偏差1の正規乱数を10000個発生
# ヒストグラムの描画
hist(z, freq = FALSE, # 密度スケールで表示
     breaks = 50) # ビンの数を約50個に設定
curve(dnorm(x, 1, 1), add = TRUE, col = "red") # 理論上の確率密度関数を上書き
## ガンマ分布
y <- rgamma(10000, shape = 8, rate = 4) # 形状4, レート4のガンマ分布に従う乱数を10000個発生
hist(y, freq = FALSE, breaks = 50, col = "skyblue", border = "skyblue")
curve(dgamma(x, shape = 8, rate = 4), add = TRUE, 
      col = "royalblue", lwd = 3) # 理論上の確率密度関数を上書き

### 度数分布表
## ポアソン分布
N <- rpois(10000, lambda = 10) # 強度10のポアソン分布に従う乱数を10000個発生
(tab <- table(N)) # 度数分布表
plot(tab/10000, ylab = "Relative frequency") # 相対度数をグラフ化
## 確率質量関数の上書き
idx <- min(N):max(N)
points(idx, dpois(idx, lambda = 10), pch = 4, 
       cex = 1.5, col = "orange")

#################################################
#  制御文
#################################################

### 条件分岐：if文
## xが0以上だったらxの平方根を返し, 0未満だったら
## "xは負です"と表示する関数の作成
f <- function(x){
  if(x >= 0){ # x >= 0がTRUEの場合の操作を記述
    return(sqrt(x))
  }else{ # x >= 0がFALSEの場合の操作を記述
    print("xは負です")
  }
}
f(4) # 2
f(-1) # "xは負です"
## else以下はなくてもよい
g <- function(x){
  if(x >= 0){ # x >= 0がTRUEの場合の操作を記述
    return(sqrt(x))
  }
}
g(4) # 2
g(-1) # 何も起きない

### 繰り返し：for文
## ランダムウォーク
## X(1)=1, X(n+1)=X(n)+e(n) (n=1,2,...,100)
## を作成する. ただし, e(1),e(2),...は標準正規乱数
set.seed(123) # 乱数シードを固定
X <- double(100) # 計算結果を保持するために, 0を100個並べたベクトルを用意しておく
X[1] <- 1 # 初期値
for(n in 2:100){ # nをベクトル2:100の成分すべてを順番に動かしながら, 以下の処理を実行
  X[n] <- X[n-1] + rnorm(1)
}
plot(X, type = "o", col = "red") # プロット
## 実は, 上の操作は関数diffinvを使うことで容易に実行可能
set.seed(123) # 乱数シードをリセット
Y <- diffinv(rnorm(99), # 差分をとるとrnorm(99)になるベクトルを作成
             xi = 1) # ベクトルの初期値(デフォルトは0)
lines(Y, type = "p", pch = 19, col = "blue") # 上書き(完全に一致する)

### 関数replicateの使い方
## 同じプログラムを何度も繰り返した結果を得るのに便利
## モンテカルロ・シミュレーションで重宝する

## 例1
## 「n個の標準正規乱数の標本平均にsqrt(n)をかけて標準偏差
## で割った統計量」を10000個計算する例を示す
# 「n個の標準正規乱数の標本平均にsqrt(n)をかけて標準偏差
# で割った統計量」を計算する関数の作成
myf <- function(n){ 
  x <- rnorm(n) # 乱数の発生
  xbar <- mean(x) # 標本平均
  s <- sd(x) # 標準偏差
  return(sqrt(n)*xbar/s)
}
# モンテカルロ・シミュレーション
# myf(10)を10000回実行する
result <- replicate(10000, myf(10))
# ヒストグラムを描画して理論上の確率密度関数を上書きしてみる
hist(result, freq = FALSE, breaks = 50) 
curve(dt(x, df = 9), add = TRUE, col = "red") # 自由度9のt分布の確率密度関数を上書き

## 例2
## for文のところで作成したランダムウォークを1000個発生させる
rw <- function(){ # ランダムウォークを作る関数
  return(diffinv(rnorm(99), xi = 1))
}
# モンテカルロ・シミュレーション
# rw()を1000回実行する
result <- replicate(1000, rw())
str(result) # 100 x 1000行列
matplot(result, type = "l", lty = 1, 
        col = "gray") # 1000個のパスを同時に描画
