
###################################################
#  quantmodパッケージによる金融時系列データの取得
###################################################

install.packages("quantmod") # パッケージのインストール
library(quantmod) # パッケージのロード

### 株価データの取得
## マイクロソフトの2016年1月1日から2017年12月31日までの株価データを取得する
## マイクロソフトに対応するシンボルは"MSFT"
## データの取得元はデフォルトではyahoo financeであり, 銘柄を表すシンボルは
## yahoo financeで利用されているものに対応
## https://finance.yahoo.com から検索できる
getSymbols("MSFT", from = "2016-01-01", to = "2017-12-31")
str(MSFT) 
# Yahoo Financeからダウンロードされたデータがxts(extended time series)
# オブジェクトMSFTに変換されている
head(MSFT) # 最初の6行を表示
plot(MSFT$MSFT.Close) # 終値をプロット

### 株価データの取得: 日本株の場合
## トヨタ自動車の2016年1月1日から2017年12月31日までの株価データを取得する
## トヨタ自動車に対応するシンボルは"7203.T"
getSymbols("7203.T", from = "2016-01-01", to = "2017-12-31")
str(7203.T) # エラーが出る(数字から始まる変数名はRでは扱えない)
toyota <- get("7203.T") # これで取得できる
str(toyota)
head(toyota)
plot(toyota[ ,"7203.T.Close"]) # 終値をプロット
# plot(toyota$"7203.T.Close") # $を使う場合

###################################################
#  zooパッケージによる時系列データの扱い
###################################################

## yuimaパッケージでは, 基本的に時系列データ(観測値+観測時点)を
## zooパッケージの提供するzooオブジェクトという形式で保持している

install.packages("zoo") # パッケージのインストール
library(zoo) # パッケージのロード

### 上で取得したトヨタの株価データをzooに変換してみる
## 方法1: as.zooを使う
## Rには時系列データを扱うためのクラスが数多く実装されているが,
## 大抵それらはas.zooによってzooに変換できる
toyota1 <- as.zoo(toyota) # zooに変換
str(toyota1)
idx <- time(toyota1) # 観測時点の取得
head(idx) # 最初の6個
head(index(toyota1)) # 上と同じ
plot(toyota1) # プロット(6個の時系列すべて)
plot(toyota1[ ,"7203.T.Open"]) # プロット(終値)

## 注: xtsは観測時点の重複を想定しているが, zooは
## 想定していない. そのため, 重複する観測時点を
## 含むxtsオブジェクトをzooに変換すると警告がでる
## (実行はできる)

## 方法2: 観測値と観測時点を明示的に指定
obs <- as.matrix(toyota) # 観測値
idx <- time(toyota) # 観測時点
toyota2 <- zoo(obs, idx) # zooオブジェクトの作成
str(toyota2) # toyota1と同じ
plot(toyota2)
